/*==============================================================================
PART 2D: Mark-to-Market Asset Loss Calculation
==============================================================================
Purpose:
This script calculates mark-to-market losses for bank assets, specifically
focusing on securities and loans categorized into maturity buckets. It uses
Bloomberg index price changes to estimate potential losses based on the
composition of bank balance sheets. Refer to the manuscript for details on the
methodology and specific calculations.

Input:
- $path_clean/call_reports_prepared.dta (Prepared bank-quarter panel from Part 1)
- $path_raw/indices_price_oad.dta or $path_temp/indices_price_oad${ir_suffix}.dta:
  Bloomberg index data including price changes (dp_*, os_dp_*, f1_os_dp_*) for
  various Treasury and MBS indices.
- $path_temp/deposit_franchises${ext_suffix}.dta (Estimated deposit franchises from Part 2C)
  (Conditionally merged for capital requirements extensions)

Output:
- A Stata dataset in long panel form containing bank-period observations.
  Each observation identifies `rssdid`, `yq` (the end quarter of the period),
  and `period` (the name of the estimation period). It includes the estimated
  mark-to-market asset loss (`loss`) for that specific period, expressed as a
  percentage of base quarter assets.
  Example structure (simplified):
     +-----------------------------------------------------------------+
     | rssdid       yq      period         loss   tot_assets |
     |-----------------------------------------------------------------|
  1. |   6329   2019q2   dec2021     -1.500000   10000000 |
  2. |   6329   2022q4   feb2023     -5.250000   10500000 |
  3. |   6329   2023q4   feb2024     -8.100000   11000000 |
  4. |   6330   2019q2   dec2021     -1.200000    5000000 |
  5. |   6330   2022q4   feb2023     -4.800000    5200000 |
     |-----------------------------------------------------------------|
  6. |   6330   2023q4   feb2024     -7.500000    5500000 |
     ......continued .....


Methodology:
1. Define broad asset categories by maturity (non-mortgage and mortgage) by summing
   holdings across detailed balance sheet items and maturity buckets.
2. Merge in Bloomberg index price change data & map relevant index price changes
   to the defined asset buckets.
3. Construct bank-specific weighted index for "other MBS" based on bank holdings
   across finer RMBS maturity buckets (see Appendix C.2. for details).
4. Calculate estimated losses for each asset bucket by multiplying the
   bucket's book value (from a base quarter, primarily 2021q4) by the
   corresponding index price change over the given period.
5. Handle cases where a bank does not exist in the primary base quarter
   (2021q4) by using an alternative base quarter (2022q1).
6. Aggregate losses across all buckets to get total estimated asset loss.
7. Adjust losses for capital requirements extensions if specified.
8. Express the total asset loss as a percentage of total assets from the
   primary base quarter (2021q4 or 2022q1).

Last updated: July 25, 2025
==============================================================================*/
display "--- Starting Part 2D: Mark-to-Market Asset Loss Calculation ---"

clear all // Clear memory of any previous dataset
use "$path_clean/call_reports_prepared.dta", clear // Load the prepared bank-quarter panel data

sort rssdid yq
tsset rssdid yq // Declare data as panel

//===============================================================================
// Step 1: Define Broad Asset Categories
//===============================================================================
// Purpose: Create variables representing the total book value of assets within
// broad non-mortgage and mortgage buckets, based on detailed balance sheet items
// and maturity buckets reported in Call Reports.
// See Table A.4 in the paper for the bucket classification.

// Define a variable for total assets used in the denominator later for scaling losses.
// This includes assets used for duration calculation (loans + all securities incl. "other MBS") plus cash and fed funds.
// See 1b_asset_duration.do for definition of asset_dur_tot2.
gen tot_assets = asset_dur_tot2 + cash + fedfundsrepoasset

// Define non-mortgage asset buckets by summing relevant non-residential loan and Treasury holdings by maturity
gen nonmortbucket1 = nonresloans_less_3m + nonresloans_3m_1y + securitiestreasury_less_3m + securitiestreasury_3m_1y // less than 1 year maturity
gen nonmortbucket2 = nonresloans_1y_3y + securitiestreasury_1y_3y // 1y to 3y maturity
gen nonmortbucket3 = nonresloans_3y_5y + securitiestreasury_3y_5y // 3y to 5y maturity
gen nonmortbucket4 = nonresloans_5y_15y + securitiestreasury_5y_15y // 5y to 15y maturity
gen nonmortbucket5 = nonresloans_over_15y + securitiestreasury_over_15y // over 15y maturity

// Define mortgage asset buckets by summing relevant residential loan and RMBS holdings by maturity
gen mortbucket1 = resloans_less_3m + resloans_3m_1y + securitiesrmbs_less_3m + securitiesrmbs_3m_1y // less than 1 year maturity
gen mortbucket2 = resloans_1y_3y + resloans_3y_5y + securitiesrmbs_1y_3y + securitiesrmbs_3y_5y // 1y to 5y maturity
gen mortbucket3 = resloans_5y_15y + securitiesrmbs_5y_15y // 5y to 15y maturity
gen mortbucket4 = resloans_over_15y + securitiesrmbs_over_15y // over 15y maturity
gen mortbucket5 = securitiesothermbs_less_3y // Other MBS less than 3 years maturity
gen mortbucket6 = securitiesothermbs_over_3y // Other MBS 3 years and over maturity

//===============================================================================
// Step 2: Merge & Prepare Index Price Data
//===============================================================================
////* 2a: Merge Bloomberg Index Price Data
// Purpose: Bring in the necessary Bloomberg index price change data to calculate
// mark-to-market losses.
// Input: Bloomberg index price change data (indices_price_oad.dta) containing
// price changes for various Treasury and MBS indices for all quarters 2015q1 to 2023q4.

// Merge in Bloomberg index data. The file location depends on whether an extension suffix is used.
if "${ir_suffix}" == "" | "${ir_suffix}" == "." { // If no suffix or default suffix, use raw data
    merge m:1 yq using "$path_raw/indices_price_oad", keepusing(p_* os_dp* dp* f1*) nogen // Merge by quarter, keeping only necessary price change variables
}
else { // Otherwise, use data from the temporary directory (for extensions)
    merge m:1 yq using "$path_temp/indices_price_oad${ir_suffix}", keepusing(p_* os_dp* dp* f1*) nogen // Merge by quarter, keeping only necessary price change variables
}

// Ensure data remains sorted and time-series set after merge
sort rssdid yq
tsset rssdid yq

// Filter the dataset to keep only the specific quarters relevant for the analysis periods (end quarters + alternative base quarter)
drop if !inlist(yq, tq(2021q4), tq(2022q1), tq(2022q4), tq(2023q4))

////// 2b: Construct Bank-Specific Weighted Index for "Other MBS" Buckets
// Purpose: Bloomberg indexes and Call Reports have different maturity buckets for
// "other MBS" securities. This section constructs a bank-specific synthetic price
// change index for the Call Report "other MBS" buckets (<3y and >=3y).
// Methodology: We create bank-specific weights based on the bank's holdings
// across finer *RMBS* maturity buckets (as reported in Call Reports). These weights
// are then applied to the corresponding Bloomberg *Treasury* and *MBS* index price
// changes to calculate a weighted-average price change for the Call Report "other MBS" buckets.
// See Appendix C.2. in the paper for detailed formulas and index mapping.

// Calculate weights for "other MBS" sub-categories less than 3 years maturity, based on RMBS holdings
foreach v in "less_3m" "3m_1y" "1y_3y"{
    gen othermbs_w_`v' = (securitiesrmbs_`v') / (securitiesrmbs_less_3m + securitiesrmbs_3m_1y + securitiesrmbs_1y_3y)
    // Replace missing weights with 0 if the denominator is zero or missing
    replace othermbs_w_`v' = 0 if mi(othermbs_w_`v')
}

// Calculate weights for "other MBS" sub-categories 3 years and over maturity, based on RMBS holdings
foreach v in "3y_5y" "5y_15y" "over_15y"{
    gen othermbs_w_`v' = (securitiesrmbs_`v') / (securitiesrmbs_3y_5y + securitiesrmbs_5y_15y + securitiesrmbs_over_15y)
    // Replace missing weights with 0 if the denominator is zero or missing
    replace othermbs_w_`v' = 0 if mi(othermbs_w_`v')
}

// Calculate weighted average price changes (index) for Call Report "other MBS" buckets (<3y and >=3y)
// using the calculated weights and corresponding Bloomberg index price changes.
// The loop iterates through different price change horizons (os_d, f1_os_d, d).
foreach t in "os_d" "f1_os_d" "d"{
    // Weighted average price change for "other MBS" less than 3 years maturity
    // Uses weights from RMBS <3m, 3m-1y, 1y-3y applied to Treasury 0y-1y and MBS 1y-5y indices.
    gen `t'p_othermbs_less_3y = (othermbs_w_less_3m + othermbs_w_3m_1y) * `t'p_tre_0y_1y + othermbs_w_1y_3y * `t'p_mbs_1y_5y
    // Weighted average price change for "other MBS" 3 years and over maturity
    // Uses weights from RMBS 3y-5y, 5y-15y, >15y applied to MBS 1y-5y, 15y-FNMA, and 30y indices.
    gen `t'p_othermbs_over_3y = othermbs_w_3y_5y * `t'p_mbs_1y_5y + othermbs_w_5y_15y * `t'p_mbs_15y_fnma + othermbs_w_over_15y * (2/3 * `t'p_mbs_15y_fnma + 1/3 * `t'p_mbs_30y)
}


//===============================================================================
// Step 3: Map Price Changes to Asset Buckets
//===============================================================================
// Purpose: Assign the relevant Bloomberg index price changes (including the
// constructed "other MBS" indices) to the defined asset buckets.
// See formulas and index mappings in Table A.5 of the paper.

// Loop through different price change horizons (os_, f1_os_, empty for 'd')
foreach i in "os_" "f1_os_" ""{
    // Map price changes for non-mortgage loans & securities buckets
    gen nonmortbucket1_`i'dp = `i'dp_tre_0y_1y
    gen nonmortbucket2_`i'dp = `i'dp_tre_1y_3y
    gen nonmortbucket3_`i'dp = `i'dp_tre_3y_5y
    gen nonmortbucket4_`i'dp = 4/15 * `i'dp_tre_5y_7y + 2/5 * `i'dp_tre_7y_10y + 1/3 * `i'dp_tre_10y_20y // Weighted average for this bucket
    gen nonmortbucket5_`i'dp = `i'dp_tre_10y_20y

    // Map price changes for Mortgage loans & RMBS buckets
    gen mortbucket1_`i'dp = `i'dp_tre_0y_1y // Uses Treasury index for short-term mortgage assets
    gen mortbucket2_`i'dp = `i'dp_mbs_1y_5y
    gen mortbucket3_`i'dp = `i'dp_mbs_15y_fnma
    gen mortbucket4_`i'dp = 2/3 * `i'dp_mbs_15y_fnma + 1/3 * `i'dp_mbs_30y // Weighted average for this bucket

    // Map price changes for Other MBS buckets using the constructed weighted indices
    gen mortbucket5_`i'dp = `i'dp_othermbs_less_3y // Uses the calculated weighted price change for <3y other MBS
    gen mortbucket6_`i'dp = `i'dp_othermbs_over_3y // Uses the calculated weighted price change for >=3y other MBS
}


//===============================================================================
// Step 4: Calculate Estimated Losses for Each Bucket (Base: 2021q4)
//===============================================================================
// Purpose: Estimate the mark-to-market loss for each asset bucket by multiplying
// the book value of the bucket in the primary base quarter (2021q4) by the relevant
// price change over the specific horizon (Dec 2021 to Feb 2023/Feb 2024).


// Calculate losses for each bucket relative to 2021q4 book values.
// This loop:
// 1. Creates a fixed value (`_fv`) variable for each bucket based on its book value in 2021q4.
// 2. Propagates this 2021q4 book value across all quarters for each bank.
// 3. Initializes the loss variable (`_lossq`) to 0 for the base quarter (2021q4).
// 4. Calculates the mark-to-market loss for subsequent quarters (2022q4, 2023q4)
//    by multiplying the 2021q4 book value (`_fv`) by the appropriate price change index (`_f1_os_dp`).
foreach v in "nonmort" "mort" {
    forvalues i=1/6{
        if "`v'" == "nonmort" & `i' > 5 {
            continue // Skip non-mortgage bucket 6 as it does not exist
        }

        // Store the book value from 2021q4 for each bucket
        gen `v'bucket`i'_fv = `v'bucket`i' if yq==tq(2021,4)

        // Propagate the 2021q4 book value across all quarters for each bank
        by rssdid: egen temp_`v'bucket`i'_fv = max(`v'bucket`i'_fv)
        replace `v'bucket`i'_fv = temp_`v'bucket`i'_fv
        drop temp_`v'bucket`i'_fv

        // Initialize loss variable for the base quarter (2021q4) and calculate for other quarters
        // Uses the `_f1_os_dp` price change, which represents the change from Dec 2021.
        gen `v'bucket`i'_lossq = 0 if yq==tq(2021,4)

        // Calculate loss for quarters after 2021q4 using the 2021q4 book value and price change from Dec 2021
        replace `v'bucket`i'_lossq = `v'bucket`i'_f1_os_dp * `v'bucket`i'_fv if yq > tq(2021,4)

    }
}

// Propagate the 2021q4 book value of total assets to use as reference
// for the total asset loss calculation later.
gen assets_fv = assets if yq==yq(2021,4)
bys rssdid: egen temp_assets_fv = max(assets_fv)
replace assets_fv = temp_assets_fv
drop temp_assets_fv

//===============================================================================
// Step 6: Handle Banks Missing in Primary Base Quarter (2021q4)
//===============================================================================
// Purpose: For banks that do not have data in the primary base quarter (2021q4),
// use 2022q1 as the alternative base quarter for calculating asset losses.

// Handle banks missing in 2021q4 by using 2022q1 as alternative base quarter
// This loop:
// 1. Stores the book value from 2022q1 for each bucket (`_fv22q1`).
// 2. Propagates these 2022q1 book values across all quarters for each bank.
// 3. For banks where the 2021q4 book value (`_fv`) is missing, recalculates the loss
//    in 2022q4 and 2023q4 using the 2022q1 base value (`_fv22q1`) and the same
//    price change index (`_f1_os_dp`).
foreach v in "nonmort" "mort" {
    forvalues i=1/6{
        if "`v'" == "nonmort" & `i' > 5 {
            continue // Skip non-mortgage bucket 6 as it does not exist
        }
        // Store the book value from 2022q1 for each bucket
        gen `v'bucket`i'_fv22q1 = `v'bucket`i' if yq==yq(2022,1)

        // Propagate the 2022q1 book values across all quarters for each bank
        quietly by rssdid: egen temp_`v'bucket`i'_fv22q1 = max(`v'bucket`i'_fv22q1)
        quietly replace `v'bucket`i'_fv22q1 = temp_`v'bucket`i'_fv22q1
        drop temp_`v'bucket`i'_fv22q1

        // For banks missing 2021q4 data (`_fv` is missing), recalculate loss
        // using 2022q1 base values (`_fv22q1`) and the price change from Dec 2021 (`_f1_os_dp`).
        // Note: This assumes the price change from Dec 2021 is still the relevant shock,
        // but applied to the 2022q1 balance sheet composition.
        replace `v'bucket`i'_lossq = `v'bucket`i'_f1_os_dp * `v'bucket`i'_fv22q1 if mi(`v'bucket`i'_fv)

  }
}

// Handle banks missing 2021q4 total assets by using 2022q1 total assets as alternative base
replace assets_fv = assets if mi(assets_fv) & yq==yq(2022,1)
bys rssdid: egen temp_assets_fv = max(assets_fv)
replace assets_fv = temp_assets_fv
drop temp_assets_fv


//===============================================================================
// Step 7: Final Calculations and Save
//===============================================================================
// Purpose: Sum the estimated losses across all buckets, express the total loss
// as a percentage of base assets, adjust for capital requirements extensions
// if applicable, handle missing values, and save the final dataset in long format.

// Define the end quarters for the analysis periods and create the period identifier
// Note: Correcting the mapping for dec2021 to 2021q4.
local dt_2021q4 = tq(2021q4) // dec2021 period end
local dt_2022q4 = tq(2022q4) // feb2023 period end
local dt_2023q4 = tq(2023q4) // feb2024 period end

// Create a string variable to identify the period for each observation
gen period = ""
replace period = "dec2021" if yq == `dt_2021q4'
replace period = "feb2023" if yq == `dt_2022q4'
replace period = "feb2024" if yq == `dt_2023q4'

// Compute total losses across all non-mortgage and mortgage buckets
gen temp = nonmortbucket1_lossq + nonmortbucket2_lossq + nonmortbucket3_lossq + nonmortbucket4_lossq + nonmortbucket5_lossq + mortbucket1_lossq + mortbucket2_lossq + mortbucket3_lossq + mortbucket4_lossq + mortbucket5_lossq + mortbucket6_lossq

// Calculate the total estimated loss as a percentage of base quarter total assets (assets_fv)
// Multiply by -1 because 'loss' is defined as a positive value in the paper/tables.
gen loss = -temp / assets_fv * 100
// Set loss to 0 for the dec2021 period, as this is the base period for calculating changes
replace loss = 0 if period == "dec2021"

// Adjust losses in the capital requirements extensions if the global ${cap_rule} is set
gen cap_add_fv = 0 // Initialize variable for capital add-on
if "${cap_rule}" != "" {
    display "Applying capital requirements adjustment based on cap_rule: ${cap_rule}"

    // For extension 5b, we need to bring in the deposit franchises calculated in 2c.
    // Merge the deposit franchises dataset (which is in long format by rssdid and period)
    // to the current dataset (which is still in bank-quarter format but filtered to end quarters).
    // This requires preserving the current data, merging, saving to a tempfile, and restoring.
    preserve
    drop if period == "" // Drop quarters not corresponding to identified periods before merging
    merge 1:1 rssdid period using "$path_temp/deposit_franchises${ext_suffix}.dta", keep(3) nogen // Merge deposit franchises
    tempfile temp_deposit_franchises  
    save `temp_deposit_franchises', replace // Save the merged data to the tempfile
    restore // Restore the original dataset (before the preserve/drop/merge)

    // Merge the deposit franchises from the tempfile back into the main dataset using rssdid and yq
    // This brings the franchise values into the bank-quarter structure for the end quarters.
    merge 1:1 rssdid yq using `temp_deposit_franchises', keep(master match) nogen

    // Calculate the capital add-on based on the specified cap_rule global.
    // The cap rules are typically not normalized by assets or relative to 2021q4 values.
    // Here we normalize by assets and fix the value to the 2021q4 (or 2022q1 alternative) base.
    replace cap_add_fv = ${cap_rule} / assets if yq == yq(2021,4) // Calculate add-on normalized by 2021q4 assets
    bys rssdid: egen temp_cap_add_fv = max(cap_add_fv) // Propagate the 2021q4 value
    replace cap_add_fv = temp_cap_add_fv
    drop temp_cap_add_fv

    // Handle banks missing in 2021q4 by using 2022q1 as alternative base quarter for the capital add-on normalization
    replace cap_add_fv = ${cap_rule} / assets if mi(cap_add_fv) & yq == yq(2022,1) // Calculate add-on normalized by 2022q1 assets
    bys rssdid: egen temp_cap_add_fv = max(cap_add_fv) // Propagate the 2022q1 value
    replace cap_add_fv = temp_cap_add_fv
    drop temp_cap_add_fv

    // Adjust the calculated asset loss by multiplying it with (1 + cap_add_fv).
    // This incorporates the capital requirement adjustment into the loss measure.
    replace loss = loss * (1 + cap_add_fv)
}

// Replace any remaining missing loss values with the period mean.
// This handles cases where loss could not be calculated for a bank in a specific period.
quietly sum loss if yq == yq(2022,4) // Calculate mean for feb2023 period
replace loss = r(mean) if mi(loss) & yq == yq(2022,4)
quietly sum loss if yq == yq(2023,4) // Calculate mean for feb2024 period
replace loss = r(mean) if mi(loss) & yq == yq(2023,4)
// Note: Missing values for dec2021 (yq==2021q4) are not filled here, as loss is explicitly set to 0.


// Drop the 2022q1 observations as they were only needed as an alternative base quarter
drop if yq == yq(2022,1)

// Keep only relevant variables for the final long format output dataset
keep rssdid period loss

// Sort the final dataset
sort rssdid period

// Save the dataset to the temporary directory
save "$path_temp/mtm_assets${ext_suffix}.dta", replace

display "--- Mark-to-Market Asset Loss Calculation completed ---" // Indicate the completion of the script
